function AllenCahnString()
% finds transitions in the Allen-Cahn 2D system
% u_t = kappa(u_{xx} + u_{yy}) + u - u^3 + \epsilon*eta,
% where eta is white noise
% BC: u(x = 0,y) = u(x = 1,y) = 1;
% u(x,y = 0) = u(x,y = -1) = -1;
% Domain = [0,1]^2;
%% setup
kappa = 0.005;
N1 = 51;
N0 = N1 - 1;
N = N1 - 2;
h = 1/N0;
kh = kappa/h^2;
NN = N*N; % the number of mesh points with unknown values
% set up boundary conditions
u = zeros(N1);
u(1,:) = -1;
u(end,:) = -1;
u(:,1) = 1;
u(:,end) = 1;
u(1,1) = 0; u(end,end) = 0; u(1,end) = 0; u(end,1) = 0;
x = linspace(0,1,N1);
Imid = 2 : N0;
% set up Laplacian matrix 
I = eye(N);
e = ones(N,1);
T = spdiags([e,-4*e,e],[-1:1],N,N);
Aux = spdiags([e,e],[-1,1],N,N);
L = kh*(kron(I,T) + kron(Aux,I));
% set up boundary term
b = zeros(NN,1);
b(1 : N) = 1;
b(NN - N + 1 : NN) = 1;
b(1 : N : end) = b(1 : N : end) - 1;
b(N : N : end) = b(N : N : end) - 1;
b = b*kh;
% set up the right-hand rise of the Allen-Cahn PDE
F = @(x)L*x + b + x - x.^3; 
%% find equilibria
vguess = ones(NN,1);
v0 = fsolve(F,vguess);
v1 = fsolve(F,-vguess);
u0 = u;
u0(Imid,Imid) = reshape(v0,[N,N]);
u1 = u;
u1(Imid,Imid) = reshape(v1,[N,N]);

figure(1); clf; hold on; 
imagesc(x,x,u0);
axis tight
daspect([1,1,1]);
set(gca,'Fontsize',20);
xlabel('x','Fontsize',20);
ylabel('y','Fontsize',20);

figure(2); clf; hold on; 
imagesc(x,x,u1);
axis tight
daspect([1,1,1]);
set(gca,'Fontsize',20);
xlabel('x','Fontsize',20);
ylabel('y','Fontsize',20);

%% the string method
tol = 1e-3;
shift = Inf;
% initial path
Nim = 39;
dn = 2;
Nim1 = Nim - 1;
g1 = linspace(0,1,Nim)';
path = interp1([0;1],[v0,v1]',g1);
figure(3); clf;
for j = 1 : dn : Nim
    subplot(4,5,(j-1)/dn + 1);
    uu = u;
    uu(Imid,Imid) = reshape(path(j,:)',[N,N]);
    imagesc(x,x,uu);
    daspect([1,1,1]);
end
s = 0.01;
step = 0;
while shift > tol
    % gradient descent
    p_old = path;
    for j = 2 : Nim1
        f = F(path(j,:)')';
        path(j,:) = path(j,:) + s*f;
    end
    %reparametrization
    dp = path - circshift(path,1,1);
    dp(1,:) = 0;
    dl = sqrt(sum(dp.^2,2));
    lp = cumsum(dl);
    len = lp(end);
    lp = lp/len; % normalize
    path = interp1(lp,path,g1);
    shift = norm(path - p_old);
    step = step + 1;
    fprintf('step = %d, shift = %d\n',step,shift);
end
figure(4); clf;
for j = 1 : dn : Nim
    subplot(4,5,(j-1)/dn + 1);
    uu = u;
    uu(Imid,Imid) = reshape(path(j,:)',[N,N]);
    imagesc(x,x,uu);
    daspect([1,1,1]);
end
%% plot energy along the found path
% V = 0.5*iint_{Omega} kappa*(u_x^2 + u_y^2) + (1 - u^2)^2
kh1 = 0.5*kappa/h;
Dy = kh1*kron(I,spdiags([-e,e],[-1,1],N,N));
A1 = spdiags([-e,e],[-1,1],N,N);
Dx = kh1*kron(A1,I);
bx = zeros(NN,1);
bx(1 : N) = -1;
bx(NN - N + 1 : end) = 1;
bx = bx*kh1;
by = zeros(NN,1);
by(1 : N : end) = 1;
by(N : N : end) = -1;
by = by*kh1;
Energy = @(x)sum(sum(0.5*(Dx*x + bx).^2 + (Dy*x + by).^2 + 0.25*(1 - x.^2).^2))*h^2;
figure(5); clf; hold on; grid;
E = zeros(Nim,1);
for j = 1 : Nim
    E(j) = Energy(path(j,:)');
end
plot(g1,E,'Linewidth',2,'Marker','.','Markersize',20);
set(gca,'Fontsize',20);
xlabel('Parameter along the path','Fontsize',20);
ylabel('Energy','Fontsize',20);
end


