function NEB_demo_2well()
% implements the Nudged Elastic Band method
% JOURNAL OF CHEMICAL PHYSICS VOLUME 113, NUMBER 22 8 DECEMBER 2000
% Improved tangent estimate in the nudged elastic band method for 
% finding minimum energy paths and saddle points
% Graeme Henkelman and Hannes Jonsson

Nim = 21; % the number of images along the path including the endpoints
Nim1 = Nim - 1;
iter_max = 2000; 
xa = [-1, -1];
xb = [1, -1];
N = length(xa);

[xx,yy] = meshgrid(linspace(-2,2,200));
v = mypot(xx,yy);

figure;
hold on;
contourf(xx,yy,v,[-2:0.1:5]);

x=linspace(xa(1),xb(1),Nim)';
y=linspace(xa(2),xb(2),Nim)';

curve = plot(x,y,'w.','Markersize',20);
set(gca,'Fontsize',20);
xlabel('x','Fontsize',20);
ylabel('y','fontsize',20);
daspect([1,1,1])

drawnow;

t = linspace(0,1,Nim);
xpath = interp1([0 1],[xa; xb],t); % rows of xpath represent images

[pot,g] = mygrad(xpath);

iter_min = 10;
iter = 0;

% parameters for FIRE
dt = 1e-2;

vpath = zeros(size(xpath(2:Nim1,:)));

sstep = 0; % the number of steps in a row for which P > 0
for iter = 1 : iter_max
    Vmax(iter) = max(pot);    
    if iter > iter_min & abs(Vmax(iter - 10) - Vmax(iter)) < 1e-4
        fprintf('iter = %d\t max(V) = %d\n',iter,max(pot));
        break
    end
    tau = unit_tangent_vector(xpath,pot);
    Sp = spring_force(xpath,tau);
    gperp = g(2:Nim1,:) - sum(g(2:Nim1,:).*tau,2)*ones(1,N).*tau;
    f = - gperp + sum(Sp.*tau,2)*ones(1,N).*tau; % force acting on the images
    fnor = norm(f(:));
    % update path
    xpath(2:Nim1,:) = xpath(2:Nim1,:) + dt*f; %vpath;
    set(curve,'xdata',xpath(:,1),'ydata',xpath(:,2)) 
    drawnow
    [pot,g] = mygrad(xpath);
end 
[Vmax imax] = max(pot);
if sum(isnan(pot)) == 0 & sum(isinf(pot)) == 0 & size(tau,1) > imax & imax > 1
    x0 = xpath(imax,:);
    tau0 = tau(imax - 1,:);
    x1 = x0 -tau0*0.1;
    x2 = x0 + tau0*0.1;
    fprintf('NEB: iter = %d\t max(V) = %d\n',iter,Vmax);
    flag = 1;
else
    fprintf('NEB failed, imax = %d\n',imax);
    flag = 0;
    x1 = 0;
    x2 = 0;
end
end
    

%%
function tau = unit_tangent_vector(p,pot)
[Nim,N] = size(p);
% returns tau of size [Nim - 2, N]. Does not compute unit tangent vector at
% the endpoints
% uses the method from 
% JOURNAL OF CHEMICAL PHYSICS VOLUME 113, NUMBER 22 8 DECEMBER 2000
% Improved tangent estimate in the nudged elastic band method for 
% finding minimum energy paths and saddle points
% Graeme Henkelman and Hannes Jonsson
Nim1 = Nim - 1;
Nim2 = Nim - 2;
tau = zeros(Nim2,N);
tplus = p(3 : Nim,:) - p(2 : Nim1,:);
tminus = p(2 : Nim1,:) - p(1 : Nim2,:);
dVplus = pot(3 : Nim) - pot(2 : Nim1);
dVminus = pot(2 : Nim1) - pot(1 : Nim2);
dVplus_sign = sign(dVplus);
dVminus_sign = sign(dVminus);
dVsign = dVplus_sign + dVminus_sign;
iplus = find(dVsign == 2);
iminus = find(dVsign == -2);
dV0sign = sign(pot(3 : Nim) - pot(1 : Nim2));
i0plus = find(dVsign == 0 & dV0sign >= 0);
i0minus = find(dVsign == 0 & dV0sign < 0);
dVmax = max(abs(dVplus),abs(dVminus));
dVmin = min(abs(dVplus),abs(dVminus));
tau(iplus,:) = tplus(iplus,:);
tau(iminus,:) = tminus(iminus,:);
tau(i0plus,:) = tplus(i0plus,:).*(dVmax(i0plus)*ones(1,N)) ...
    + tminus(i0plus,:).*(dVminus(i0plus)*ones(1,N));
tau(i0minus,:) = tplus(i0minus,:).*(dVmin(i0minus)*ones(1,N)) ...
    + tminus(i0minus,:).*(dVmax(i0minus)*ones(1,N));
taux = sqrt(sum(tau.^2,2));
ind = find(taux == 0);
tau(ind,:) = p(ind + 1,:) - p(ind - 1,:);
taux = sqrt(sum(tau.^2,2));
tau = tau./(taux*ones(1,N));
end

%%
function Sp = spring_force(p,tau)
[Nim,N] = size(p);
Nim1 = Nim - 1;
Sp = zeros(size(tau));
dp = sqrt(sum((p(2 : Nim,:) - p(1 : Nim1,:)).^2,2));
Sp = tau.*((dp(2 : Nim1) - dp(1 : Nim - 2))*ones(1,N));
end

%%
%%
%%
function v = mypot(x,y)
v = x.^4 - 2*x.^2 + (y - cos(pi*x)).^2;
end

function [pot,g] = mygrad(x)
g(:,1) = 4*(x(:,1).^3 - x(:,1)) + 2*pi*sin(pi*x(:,1)).*(x(:,2) - cos(pi*x(:,1)));
g(:,2) = 2*(x(:,2) - cos(pi*x(:,1)));
pot = x(:,1).^4 - 2*x(:,1).^2 + (x(:,2) - cos(pi*x(:,1))).^2;
end






